/*
 * Extended Operating System Loader (XOSL)
 * Copyright (c) 1999 by Geurt Vos
 *
 * This code is distributed under GNU General Public License (GPL)
 *
 * The full text of the license can be found in the GPL.TXT file,
 * or at http://www.gnu.org
 */

#include <palette.h>
#include <rgb.h>
#include <mem.h>
#include <graph.h>

#define Cli() (__emit__(0xfa))
#define Sti() (__emit__(0xfb))

extern THSLPalette DefaultPalette[256];
extern THSLPalette ColorSchemes[16][8];

CPalette::CPalette()
{
	Palette = new TPalette;
	Scheme = 15;
	Hue = 0;
	Saturation = 0;
	Alpha = 0;
	Fading = false;
	FadeColorInt = 100;
}

CPalette::~CPalette()
{
	delete Palette;
}

void CPalette::Install()
{
	CopyDefaultPalette();
	UpdatePalette();
	if (!Fading)
		InstallPalette();
	else
		memcpy(PreFade,Palette,sizeof (TRGBPalette[256]));
}

void CPalette::SetScheme(int Index)
{
	Scheme = Index;
}

void CPalette::SetHue(int Hue)
{
	this->Hue = Hue;
}

void CPalette::SetSaturation(int Saturation)
{
	this->Saturation = Saturation;
}

void CPalette::SetColorLayer(int Red, int Green, int Blue, int Alpha)
{
	ColorLayer.Red = Red;
	ColorLayer.Green = Green;
	ColorLayer.Blue = Blue;
	this->Alpha = Alpha;
}

void CPalette::CopyDefaultPalette()
{
	memcpy(Palette,DefaultPalette,sizeof (TRGBPalette[256]));
}

void CPalette::UpdatePalette()
{
	int Index;
	int RangeR, RangeG, RangeB;
	TRGBPalette TitleBar[2];


	// Install Color Scheme
	memcpy(&Palette->HSLPalette[16],ColorSchemes[Scheme],sizeof (THSLPalette[8]));

	// Update Hue and Saturation
	for (Index = 0; Index < 256; ++Index) {
		Palette->HSLPalette[Index].Hue += Hue;
		Palette->HSLPalette[Index].Hue &= 63;
		Palette->HSLPalette[Index].Saturation += Saturation;
		if (Palette->HSLPalette[Index].Saturation > 63)
			Palette->HSLPalette[Index].Saturation = 63;
		else
			if (Palette->HSLPalette[Index].Saturation < 0)
				Palette->HSLPalette[Index].Saturation = 0;
		HSL2RGB(Palette->RGBPalette[Index].Red,Palette->RGBPalette[Index].Green,
				  Palette->RGBPalette[Index].Blue,Palette->HSLPalette[Index].Hue,
				  Palette->HSLPalette[Index].Saturation,Palette->HSLPalette[Index].Luminance);
	}
	// update title bar
	memcpy(TitleBar,&Palette->RGBPalette[22],sizeof (TRGBPalette[2]));

	RangeR = TitleBar[1].Red - TitleBar[0].Red >> 1;
	RangeG = TitleBar[1].Green - TitleBar[0].Green >> 1;
	RangeB = TitleBar[1].Blue - TitleBar[0].Blue >> 1;
	for (Index = 0; Index < 18; ++Index) {
		Palette->RGBPalette[Index + 22].Red = TitleBar[0].Red + (RangeR * Index) / 17;
		Palette->RGBPalette[Index + 22].Green = TitleBar[0].Green + (RangeG * Index) / 17;
		Palette->RGBPalette[Index + 22].Blue = TitleBar[0].Blue + (RangeB * Index) / 17;
	}

	for (Index = 0; Index < 256; ++Index)
		MergeColors(Palette->RGBPalette[Index],ColorLayer,Alpha);

	// Merge with color layer
}

void CPalette::SetPaletteEntry(int Index, int Red, int Green, int Blue)
{
	SetRGB(Index,Red,Green,Blue);
}

void CPalette::LockFading()
{
	int Index;

	Fading = true;
	PreFade = new TRGBPalette[256];
	for (Index = 0; Index < 256; ++Index)
		GetRGB(Index,PreFade[Index].Red,PreFade[Index].Green,PreFade[Index].Blue);
}

void CPalette::FadeIn()
{
	int Index;
	int Red, Green, Blue;
	int Progress;

	Cli();
	for (Progress = 0; Progress < 63; ++Progress) {
		if ((Progress % 3) == 0)
			Graph->WaitRetrace();
		for (Index = 0; Index < 256; ++Index) {
			GetRGB(Index,Red,Green,Blue);
			if (Red != PreFade[Index].Red)
				if (Red < PreFade[Index].Red)
					++Red;
				else
					--Red;
			if (Green != PreFade[Index].Green)
				if (Green < PreFade[Index].Green)
					++Green;
				else
					--Green;
			if (Blue != PreFade[Index].Blue)
				if (Blue < PreFade[Index].Blue)
					++Blue;
				else
					--Blue;
			SetRGB(Index,Red,Green,Blue);
		}
	}
	Sti();
	delete PreFade;
	Fading = false;
}

void CPalette::FadeOut()
{
	int Red, Green, Blue;
	int Index, Progress;
	int Color;

	Color = (63 * FadeColorInt) / 100;
	Cli();
	for (Progress = 0; Progress < 63; ++Progress) {
		if ((Progress % 3) == 0)
			Graph->WaitRetrace();
		for (Index = 0; Index < 256; ++Index) {
			GetRGB(Index,Red,Green,Blue);
			if (Red != Color)
				if (Red < Color)
					++Red;
				else
					--Red;
			if (Green != Color)
				if (Green < Color)
					++Green;
				else
					--Green;
			if (Blue != Color)
				if (Blue < Color)
					++Blue;
				else
					--Blue;
			SetRGB(Index,Red,Green,Blue);
		}
	}
	Sti();
}

void CPalette::UltraFadeIn()
{
	int Index;

	Graph->WaitRetrace();
	for (Index = 0; Index < 256; ++Index)
		SetRGB(Index,PreFade[Index].Red,PreFade[Index].Green,PreFade[Index].Blue);
	delete PreFade;
	Fading = false;
}
void CPalette::UltraFadeOut()
{
	int Index;
	int Color;

	Color = (63 * FadeColorInt) / 100;
	Graph->WaitRetrace();
	for (Index = 0; Index < 256; ++Index)
		SetRGB(Index,Color,Color,Color);
}

void CPalette::FadeToBlack()
{
	int Index;
	int Progress;
	int Red, Green, Blue;

	for (Progress = 0; Progress < 63; ++Progress) {
		if ((Progress & 7) == 0)
			Graph->WaitRetrace();
		for (Index = 0; Index < 256; ++Index) {
			GetRGB(Index,Red,Green,Blue);
			if (--Red < 0)
				Red = 0;
			if (--Green < 0)
				Green = 0;
			if (--Blue < 0)
				Blue = 0;
			SetRGB(Index,Red,Green,Blue);
		}
	}
}

void CPalette::SetFadeOutColor(int Intensity)
{
	FadeColorInt = Intensity;
}

void CPalette::InstallPalette()
{
	int Index;

	Graph->WaitRetrace();
	for (Index = 0; Index < 256; ++Index)
		SetRGB(Index,Palette->RGBPalette[Index].Red,
				 Palette->RGBPalette[Index].Green,Palette->RGBPalette[Index].Blue);
}

void CPalette::HSL2RGB(int &Red, int &Green, int &Blue, int Hue, int Saturation, int Luminance)
{
	int Temp1, Temp2;
	int RedTemp, GreenTemp, BlueTemp;

	// 1
	if (Saturation == 0)
		Red = Green = Blue = Luminance;
	else {
		// 2
		if (Luminance < 32)
			Temp2 = (Luminance * (63 + Saturation)) / 63;
		else
			Temp2 = Luminance + Saturation - (Luminance * Saturation) / 63;

		// 3
		Temp1 = 2 * Luminance - Temp2;

		// 4
		// Not necessary

		// 5

		RedTemp = Hue + 21;
		GreenTemp = Hue;
		BlueTemp = Hue - 21;

		if (RedTemp > 63)
			RedTemp -= 64;
		if (BlueTemp < 0)
			BlueTemp += 64;

		// 6
		Red = GetColor(Temp1, Temp2, RedTemp);
		Green = GetColor(Temp1, Temp2, GreenTemp);
		Blue = GetColor(Temp1, Temp2, BlueTemp);
	}

	// 7
	// Not necessary
}

int CPalette::GetColor(long Temp1, long Temp2, long Temp3)
{
	if (Temp3 < 11)
		return Temp1 + ((Temp2 - Temp1) * 378 * Temp3) / 3969;
	if (Temp3 < 32)
		return Temp2;
	if (Temp3 < 42)
		return Temp1 + ((Temp2 - Temp1) * (42 - Temp3) * 378) / 3969;
	return Temp1;
}

void CPalette::MergeColors(TRGBPalette &Back, const TRGBPalette &Front, int Alpha)
{
	int iAlpha;

	if (!Alpha)
		return;
	iAlpha = 64 - Alpha;
	Back.Red = (Back.Red * iAlpha + Front.Red * Alpha) >> 6;
	Back.Green = (Back.Green * iAlpha + Front.Green * Alpha) >> 6;
	Back.Blue = (Back.Blue * iAlpha + Front.Blue * Alpha) >> 6;
}

THSLPalette DefaultPalette[256] = {
{0x00,0x00,0x00},{0x2a,0x3f,0x15},{0x15,0x3f,0x15},{0x1f,0x3f,0x15},
{0x00,0x3f,0x15},{0x36,0x3f,0x15},{0x05,0x3f,0x15},{0x00,0x00,0x2a},
{0x00,0x00,0x15},{0x2a,0x3f,0x2a},{0x15,0x3f,0x2a},{0x1f,0x3f,0x2a},
{0x00,0x3f,0x2a},{0x36,0x3f,0x2a},{0x0a,0x3f,0x2a},{0x00,0x00,0x3f},
{0x1f,0x3f,0x0f},{0x00,0x00,0x00},{0x00,0x00,0x1f},{0x00,0x00,0x2f},
{0x00,0x00,0x37},{0x00,0x00,0x3f},{0x2a,0x3f,0x0f},{0x2a,0x3f,0x0e},
{0x2a,0x3f,0x0d},{0x2a,0x3f,0x0c},{0x2a,0x3f,0x0c},{0x2a,0x3f,0x0b},
{0x2a,0x3f,0x0a},{0x2a,0x3f,0x09},{0x2a,0x3f,0x08},{0x2a,0x3f,0x07},
{0x2a,0x3f,0x06},{0x2a,0x3f,0x06},{0x2a,0x3f,0x05},{0x2a,0x3f,0x04},
{0x2a,0x3f,0x03},{0x2a,0x3f,0x02},{0x2a,0x3f,0x01},{0x2a,0x3f,0x00},
{0x00,0x00,0x0a},{0x00,0x00,0x3a},{0x00,0x00,0x2e},{0x00,0x00,0x1e},
{0x00,0x00,0x30},{0x00,0x00,0x2c},{0x00,0x00,0x32},{0x00,0x00,0x28},
{0x00,0x00,0x18},{0x00,0x00,0x24},{0x00,0x00,0x14},{0x00,0x00,0x26},
{0x00,0x00,0x22},{0x00,0x00,0x2a},{0x28,0x30,0x2a},{0x00,0x00,0x34},
{0x00,0x00,0x38},{0x00,0x00,0x20},{0x2a,0x3f,0x01},{0x28,0x2c,0x24},
{0x00,0x00,0x36},{0x00,0x00,0x10},{0x00,0x00,0x0c},{0x24,0x07,0x2e},
{0x00,0x00,0x3c},{0x27,0x0a,0x26},{0x29,0x3f,0x13},{0x28,0x3f,0x0f},
{0x28,0x3c,0x23},{0x25,0x0c,0x32},{0x26,0x0e,0x2b},{0x28,0x14,0x26},
{0x2a,0x07,0x37},{0x28,0x3f,0x1a},{0x28,0x3c,0x24},{0x29,0x3f,0x0a},
{0x2a,0x08,0x2c},{0x27,0x15,0x1e},{0x28,0x16,0x2b},{0x28,0x0f,0x30},
{0x28,0x3a,0x0d},{0x2a,0x04,0x24},{0x28,0x0d,0x27},{0x28,0x31,0x12},
{0x29,0x3f,0x0d},{0x29,0x37,0x11},{0x28,0x2b,0x10},{0x2a,0x08,0x1e},
{0x28,0x3c,0x25},{0x28,0x10,0x1b},{0x27,0x09,0x24},{0x29,0x15,0x23},
{0x28,0x39,0x0b},{0x2a,0x0f,0x3b},{0x28,0x10,0x29},{0x28,0x33,0x16},
{0x28,0x1d,0x13},{0x2a,0x0a,0x39},{0x27,0x10,0x25},{0x28,0x25,0x16},
{0x24,0x07,0x12},{0x25,0x09,0x26},{0x28,0x07,0x20},{0x28,0x29,0x0e},
{0x27,0x18,0x1a},{0x28,0x14,0x29},{0x28,0x13,0x2c},{0x2a,0x06,0x35},
{0x27,0x0d,0x14},{0x28,0x15,0x28},{0x28,0x1e,0x19},{0x28,0x19,0x1c},
{0x28,0x0d,0x22},{0x28,0x27,0x12},{0x2a,0x1f,0x3d},{0x27,0x0c,0x2d},
{0x29,0x2d,0x0e},{0x28,0x23,0x14},{0x2a,0x04,0x31},{0x28,0x1e,0x15},
{0x28,0x21,0x11},{0x24,0x09,0x0e},{0x28,0x0c,0x19},{0x00,0x00,0x16},
{0x28,0x1c,0x14},{0x28,0x36,0x1c},{0x2a,0x03,0x08},{0x28,0x12,0x1c},
{0x28,0x10,0x1d},{0x28,0x17,0x18},{0x28,0x13,0x1a},{0x28,0x2a,0x18},
{0x28,0x1d,0x1e},{0x28,0x20,0x1d},{0x28,0x13,0x2f},{0x28,0x32,0x0f},
{0x28,0x2b,0x22},{0x28,0x3a,0x1b},{0x28,0x1b,0x1a},{0x27,0x0b,0x1c},
{0x28,0x16,0x2e},{0x28,0x15,0x24},{0x28,0x1b,0x0e},{0x28,0x10,0x20},
{0x28,0x05,0x1f},{0x28,0x2b,0x21},{0x28,0x31,0x1d},{0x24,0x0c,0x0a},
{0x28,0x16,0x16},{0x28,0x3f,0x07},{0x28,0x0f,0x24},{0x28,0x0a,0x1f},
{0x27,0x27,0x15},{0x2a,0x07,0x2e},{0x2a,0x0f,0x33},{0x29,0x24,0x24},
{0x28,0x2c,0x1f},{0x00,0x00,0x1c},{0x00,0x00,0x1a},{0x28,0x13,0x11},
{0x28,0x21,0x17},{0x28,0x16,0x1d},{0x29,0x1a,0x18},{0x29,0x2e,0x24},
{0x28,0x27,0x0a},{0x26,0x12,0x35},{0x2a,0x0b,0x34},{0x28,0x0e,0x21},
{0x2a,0x11,0x10},{0x28,0x16,0x20},{0x2a,0x09,0x1c},{0x28,0x2b,0x13},
{0x2a,0x0f,0x04},{0x23,0x08,0x33},{0x27,0x31,0x12},{0x28,0x37,0x08},
{0x2a,0x0f,0x15},{0x27,0x3a,0x0f},{0x27,0x0c,0x16},{0x28,0x16,0x0f},
{0x28,0x37,0x11},{0x28,0x11,0x16},{0x28,0x16,0x21},{0x24,0x06,0x2c},
{0x28,0x0e,0x2b},{0x28,0x23,0x1b},{0x27,0x08,0x22},{0x28,0x1f,0x0c},
{0x29,0x1d,0x1e},{0x26,0x1d,0x18},{0x29,0x2b,0x28},{0x2a,0x06,0x26},
{0x28,0x1b,0x24},{0x28,0x28,0x31},{0x28,0x0b,0x1b},{0x28,0x1e,0x11},
{0x2a,0x06,0x2a},{0x28,0x2c,0x2a},{0x28,0x1c,0x20},{0x28,0x3b,0x13},
{0x28,0x1c,0x2b},{0x2a,0x3f,0x02},{0x28,0x2d,0x26},{0x28,0x1a,0x2c},
{0x2a,0x02,0x21},{0x24,0x0a,0x06},{0x27,0x21,0x15},{0x28,0x26,0x19},
{0x29,0x3b,0x2b},{0x28,0x1c,0x27},{0x28,0x24,0x35},{0x29,0x1c,0x27},
{0x29,0x3c,0x29},{0x26,0x0d,0x31},{0x28,0x2a,0x2f},{0x2a,0x07,0x12},
{0x28,0x3f,0x09},{0x28,0x2e,0x17},{0x29,0x1c,0x2e},{0x28,0x2d,0x2d},
{0x26,0x18,0x17},{0x27,0x3f,0x0d},{0x24,0x05,0x18},{0x1f,0x0f,0x3b},
{0x24,0x06,0x14},{0x28,0x2e,0x07},{0x27,0x16,0x1d},{0x23,0x0c,0x37},
{0x2a,0x09,0x0e},{0x23,0x09,0x35},{0x2a,0x19,0x0d},{0x2a,0x08,0x17},
{0x24,0x07,0x10},{0x26,0x08,0x29},{0x1f,0x0a,0x39},{0x25,0x0c,0x0c},
{0x28,0x2a,0x32},{0x28,0x1c,0x23},{0x28,0x1c,0x1b},{0x28,0x2c,0x20},
{0x26,0x1f,0x39},{0x29,0x31,0x2d},{0x28,0x21,0x38},{0x29,0x29,0x1d},
{0x26,0x07,0x25},{0x28,0x10,0x13},{0x28,0x28,0x2e},{0x28,0x2c,0x35},
{0x26,0x13,0x1f},{0x2a,0x3f,0x04},{0x27,0x3f,0x0a},{0x28,0x2d,0x34},
{0x00,0x00,0x3f},{0x00,0x00,0x00},{0x00,0x00,0x12},{0x00,0x00,0x0d}
};

THSLPalette ColorSchemes[16][8] = {
{
	{0x00,0x00,0x10},{0x00,0x00,0x00},{0x22,0x12,0x1c},{0x23,0x12,0x2c},//Bidlice
	{0x22,0x12,0x31},{0x1f,0x0e,0x36},{0x24,0x13,0x1e},{0x00,0x00,0x08}
},
{
	{0x00,0x3f,0x08},{0x00,0x00,0x00},{0x07,0x0c,0x1e},{0x09,0x0e,0x2d},//Cihla
	{0x0a,0x0d,0x31},{0x0a,0x0e,0x36},{0x00,0x3f,0x10},{0x00,0x3f,0x04}
},
{
	{0x2a,0x3f,0x08},{0x00,0x00,0x00},{0x00,0x00,0x20},{0x00,0x00,0x30},//erv.,bl,modr
	{0x00,0x00,0x37},{0x00,0x00,0x3f},{0x00,0x3f,0x10},{0x00,0x3f,0x10}
},
{
	{0x1f,0x3f,0x08},{0x00,0x00,0x00},{0x00,0x00,0x20},{0x00,0x00,0x30},//rka
	{0x00,0x00,0x37},{0x00,0x00,0x3f},{0x1f,0x3f,0x10},{0x2a,0x3f,0x08}
},
{
	{0x24,0x15,0x0c},{0x00,0x00,0x00},{0x26,0x0e,0x1a},{0x24,0x0f,0x26},//Detiv den
	{0x23,0x0d,0x2c},{0x1f,0x09,0x32},{0x25,0x0e,0x1a},{0x2a,0x3f,0x08}
},
{
	{0x36,0x3f,0x08},{0x00,0x00,0x00},{0x06,0x28,0x1b},{0x06,0x15,0x28},//Dn
	{0x05,0x10,0x2a},{0x05,0x0a,0x2d},{0x07,0x27,0x1a},{0x07,0x27,0x0d}
},
{
	{0x04,0x15,0x1e},{0x00,0x00,0x00},{0x08,0x25,0x1e},{0x07,0x23,0x2f},//Javor
	{0x06,0x23,0x33},{0x05,0x24,0x38},{0x00,0x3f,0x10},{0x00,0x3f,0x1f}
},
{
	{0x2b,0x17,0x08},{0x00,0x00,0x00},{0x2a,0x18,0x20},{0x2b,0x19,0x30},//Lila
	{0x2b,0x1c,0x34},{0x2a,0x24,0x38},{0x2b,0x18,0x20},{0x2f,0x3f,0x1f}
},
{
	{0x36,0x3f,0x08},{0x00,0x00,0x00},{0x1c,0x0e,0x1a},{0x1c,0x0a,0x28},//Lilek
	{0x1d,0x0a,0x2d},{0x1f,0x09,0x32},{0x1d,0x0b,0x1b},{0x1d,0x0b,0x0d}
},
{
	{0x1d,0x12,0x0f},{0x00,0x00,0x00},{0x1d,0x18,0x1a},{0x1e,0x14,0x29},//Nmonick mod
	{0x1c,0x15,0x2e},{0x1a,0x16,0x34},{0x2a,0x3f,0x10},{0x23,0x3f,0x18}
},
{
	{0x06,0x10,0x21},{0x00,0x00,0x00},{0x07,0x10,0x20},{0x07,0x0e,0x32},//Pou
	{0x07,0x09,0x34},{0x00,0x00,0x38},{0x1f,0x3f,0x10},{0x1f,0x3f,0x04}
},
{
	{0x3e,0x09,0x24},{0x00,0x00,0x00},{0x3e,0x10,0x20},{0x3e,0x10,0x30},//Rov
	{0x3b,0x0f,0x33},{0x36,0x0e,0x36},{0x3e,0x10,0x20},{0x00,0x3f,0x08}
},
{
	{0x16,0x15,0x0c},{0x00,0x00,0x00},{0x17,0x12,0x1c},{0x16,0x10,0x2d},//Smrk
	{0x16,0x10,0x31},{0x15,0x0e,0x36},{0x16,0x12,0x1f},{0x16,0x15,0x0c}
},
{
	{0x2a,0x1c,0x24},{0x00,0x00,0x00},{0x00,0x00,0x20},{0x00,0x00,0x30},//Strom
	{0x00,0x00,0x37},{0x00,0x00,0x3f},{0x36,0x3f,0x10},{0x36,0x3f,0x08}
},
{
	{0x36,0x0e,0x0d},{0x00,0x00,0x00},{0x05,0x0a,0x18},{0x03,0x07,0x27},//vestka
	{0x06,0x08,0x2c},{0x0a,0x09,0x32},{0x2c,0x0c,0x14},{0x36,0x0e,0x0d}
},
{
	{0x1f,0x3f,0x10},{0x00,0x00,0x00},{0x00,0x00,0x20},{0x00,0x00,0x30},//Standardn XOSL
	{0x00,0x00,0x37},{0x00,0x00,0x3f},{0x2a,0x3f,0x10},{0x00,0x00,0x00}
}
};